<?php

declare(strict_types=1);

namespace Erlage\Photogram\Requests\Hashtag\Follow;

use Erlage\Photogram\Data\Query;
use Erlage\Photogram\Constants\ResponseConstants;
use Erlage\Photogram\Pattern\ExceptionalRequests;
use Erlage\Photogram\Data\Tables\Sys\RequestTable;
use Erlage\Photogram\Data\Models\Hashtag\HashtagModel;
use Erlage\Photogram\Data\Tables\Hashtag\HashtagFollowTable;
use Erlage\Photogram\Data\Models\Hashtag\Follow\HashtagFollowModel;
use Erlage\Photogram\Data\Models\Hashtag\Follow\HashtagFollowFinder;
use Erlage\Photogram\Data\Models\Hashtag\Follow\HashtagFollowBuilder;

final class HashtagFollowActions extends ExceptionalRequests
{
    /*
    |--------------------------------------------------------------------------
    | follow user
    |--------------------------------------------------------------------------
    */

    public static function add(): void
    {
        self::hashtagFollowProcessor(true);
    }

    /*
    |--------------------------------------------------------------------------
    | unfollow user
    |--------------------------------------------------------------------------
    */

    public static function remove(): void
    {
        self::hashtagFollowProcessor(false);
    }

    /*
    |--------------------------------------------------------------------------
    | follow/unfollow user
    |--------------------------------------------------------------------------
    */

    private static function hashtagFollowProcessor(bool $doFollow): void
    {
        self::process(function () use ($doFollow)
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $hashtagIdFromReq = self::$request -> findKey(HashtagFollowTable::FOLLOWED_HASHTAG_ID, RequestTable::PAYLOAD, HashtagFollowTable::TABLE_NAME);

            self::ensureValue(ResponseConstants::ERROR_BAD_REQUEST_MSG, $hashtagIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | make sure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | ensure target user exists
            |--------------------------------------------------------------------------
            */

            $targetHashtagModel = HashtagModel::findFromId_throwException($hashtagIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | if remove follow
            |--------------------------------------------------------------------------
            */

            if (false == $doFollow)
            {
                $finder = (new HashtagFollowFinder())
                    -> setFollowedHashtagId($targetHashtagModel -> getId())
                    -> setFollowedByUserId(self::$authedUserModel -> getId())
                    -> find();

                if ($finder -> isFound())
                {
                    $finder -> popModelFromResults() -> delete();
                }
            }

            /*
            |--------------------------------------------------------------------------
            | else if it's a do follow request
            |--------------------------------------------------------------------------
            */
            elseif (true == $doFollow)
            {
                /*
                |--------------------------------------------------------------------------
                | try selecting the follow first
                |--------------------------------------------------------------------------
                */

                $hashtagFollowBean = (new Query())
                    -> from(HashtagFollowTable::TABLE_NAME)
                    -> where(HashtagFollowTable::FOLLOWED_HASHTAG_ID, $targetHashtagModel -> getId())
                    -> where(HashtagFollowTable::FOLLOWED_BY_USER_ID, self::$authedUserModel -> getId())
                    -> selectOne();

                /*
                |--------------------------------------------------------------------------
                | if user is already followed
                |--------------------------------------------------------------------------
                */

                if (null != $hashtagFollowBean)
                {
                    /*
                    |--------------------------------------------------------------------------
                    | create follow model from selected bean
                    |--------------------------------------------------------------------------
                    */

                    $hashtagFollowModel = HashtagFollowModel::createFromUntouchedBean_noException($hashtagFollowBean);

                    self::ensureModel($hashtagFollowModel);

                    /*
                    |--------------------------------------------------------------------------
                    | add to follow model to follows map
                    |--------------------------------------------------------------------------
                    */

                    self::addToResponse(HashtagFollowTable::getTableName(), $hashtagFollowModel -> getDataMap());
                }

                /*
                |--------------------------------------------------------------------------
                | else if not already followed
                |--------------------------------------------------------------------------
                */

                else
                {
                    /*
                    |--------------------------------------------------------------------------
                    | build follow
                    |--------------------------------------------------------------------------
                    */

                    $hashtagFollowModel = (new HashtagFollowBuilder())
                        -> setFollowedHashtagId($targetHashtagModel -> getId())
                        -> setFollowedByUserId(self::$authedUserModel -> getId())
                        -> dispense();

                    /*
                    |--------------------------------------------------------------------------
                    | do follow
                    |--------------------------------------------------------------------------
                    */

                    $hashtagFollowModel -> save();

                    /*
                    |--------------------------------------------------------------------------
                    |  if everything is alright, add follow to follows map
                    |--------------------------------------------------------------------------
                    */

                    self::addToResponse(HashtagFollowTable::getTableName(), $hashtagFollowModel -> getDataMap());
                }
            }
        });
    }
}
